#Requires -Version 5.1

<#

Purpose:
    Display in the system tray information about the device, the same way as done with BGInfo.
    The information is retrieved using various methods, and displayed using a XAML form.
    It is available through an icon from the system tray.
    Only one instance of the script can run at any time.

Version history:
    1.0 Initial version
    
#>

# move to the script folder
Set-Location $PSScriptRoot

# Ensure only one instance runs (prevents duplicate tray icons)
$mutexName = "Global\SBFE-EUC-SystemInfoTrayApp"  # Unique name for this app
$mutex = New-Object System.Threading.Mutex($false, $mutexName)
if (-not $mutex.WaitOne(0)) {
    # Another instance is already running; exit this one
    Write-Host "Another instance is already running. Exiting."
    exit
}

# Add required assemblies
Add-Type -AssemblyName PresentationFramework, System.Drawing, System.Windows.Forms, WindowsFormsIntegration

# global data, read only once

try {
    $ComputerInfo = Get-ComputerInfo 
}
catch {
    $ComputerInfo = $null
}

# functions to retrieve device data and format it for display

function Get-LastBootTime {

    if ($ComputerInfo) {
        $result = "{0}`n{1}" -f ($ComputerInfo.OsLastBootUpTime).ToString("yyyy-MM-dd HH:mm:ss"), $ComputerInfo.TimeZone
    }
    else {
        $result = "n/a"
    }

    Write-Output $result   
}

function Get-DeviceDomain {

    if ($ComputerInfo) {
        $result = $ComputerInfo.CsDomain
    }
    else {
        $result = "n/a"
    }

    Write-Output $result   
}

function Get-IPAddress {

    $result = (Get-CimInstance -ClassName Win32_NetworkAdapterConfiguration | Where-Object { $_.IPEnabled -eq $true } | Select-Object -ExpandProperty IPAddress | Format-Table -Wrap -AutoSize -HideTableHeaders | Out-String).TrimEnd()

    Write-Output $result   
}

function Get-UserIdentity {

    $regKey = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Authentication\LogonUI"
    if (Test-Path -Path $regKey) {
        $regData = Get-ItemProperty -Path $regKey
        # user's display name
        $loggedOnDisplayName = $regData.LastLoggedOnDisplayName
        # user domain & samAccount
        $loggedOnUser = $regData.LastLoggedOnSAMUser
    }
    else {
        $loggedOnDisplayName = ""
        $loggedOnUser = ""
    }        
    try {
        # user principal name
        $userUPN = whoami.exe /upn
    }
    catch {
        $userUPN = ""
    }
    $result = (@($loggedOnDisplayName, $userUPN, $loggedOnUser) | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join "`n"

    Write-Output $result
}

function Get-OSInfo {
    try {
        $osVersion = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -ErrorAction Stop).LCUVer
        if (!$osVersion) {
            # if entry not present (Win10?) fallback to ComputerInfo data
            $osVersion = $ComputerInfo.OsVersion
        }
        if ($ComputerInfo) {
            $osName = $ComputerInfo.OsName
            $displayVersion = $ComputerInfo.OSDisplayVersion
            if (!$displayVersion) {
                # if entry not present (Win10?) fallback to registry data
                $displayVersion = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -ErrorAction Stop).DisplayVersion
            }
            $osLanguage = $ComputerInfo.OsLanguage
        }
        else {
            $osName = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -ErrorAction Stop).ProductName
            $displayVersion = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -ErrorAction Stop).DisplayVersion
            $osLanguage = ""
        }
        $result = "{0} ({1})`n{2}`n{3}" -f $osName, $displayVersion, $osVersion, $osLanguage
    }
    catch {
        $result = "n/a"
    }

    Write-Output $result
}

function Get-MSOfficeInfo {
    try {
        $updateChannel = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Inventory\Office\16.0" -ErrorAction Stop).OfficeLastUpdateChannel
        $architecture = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Inventory\Office" -ErrorAction Stop).OfficePackageArchitecture
        $packageVersion = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Inventory\Office\16.0" -ErrorAction Stop).OfficePackageVersion
        $result = "{0} ({1}) `n{2}" -f $updateChannel, $architecture, $packageVersion
    }
    catch {
        $result = "n/a"
    }

    Write-Output $result
}

function Get-BrowsersInfo {

    $browsersList = @()
    $chromePath = Join-Path ${env:ProgramFiles(x86)} -ChildPath "Google" | Join-Path -ChildPath "Chrome" | Join-Path -ChildPath "Application" | Join-Path -ChildPath "chrome.exe"
    if (Test-Path $chromePath) {
        $installedVersion = (Get-Item $chromePath).VersionInfo.ProductVersion
        $browsersList += ("Google Chrome (x86)....: {0}" -f $installedVersion)
    }
    $chromePath = Join-Path ${env:ProgramFiles} -ChildPath "Google" | Join-Path -ChildPath "Chrome" | Join-Path -ChildPath "Application" | Join-Path -ChildPath "chrome.exe"
    if (Test-Path $chromePath) {
        $installedVersion = (Get-Item $chromePath).VersionInfo.ProductVersion
        $browsersList += ("Google Chrome..........: {0}" -f $installedVersion)
    }
    $edgePath = Join-Path ${env:ProgramFiles(x86)} -ChildPath "Microsoft" | Join-Path -ChildPath "Edge" | Join-Path -ChildPath "Application" | Join-Path -ChildPath "msedge.exe"
    if (Test-Path $edgePath) {
        $installedVersion = (Get-Item $edgePath).VersionInfo.ProductVersion
        $browsersList += ("MS Edge................: {0}" -f $installedVersion)
    }
    $firefoxPath = Join-Path ${env:ProgramFiles(x86)} -ChildPath "Mozilla Firefox" | Join-Path -ChildPath "firefox.exe"
    if (Test-Path $firefoxPath) {
        $installedVersion = (Get-Item $firefoxPath).VersionInfo.ProductVersion
        $browsersList += ("Mozilla Firefox (x86)..: {0}" -f $installedVersion)
    }
    $firefoxPath = Join-Path ${env:ProgramFiles} -ChildPath "Mozilla Firefox" | Join-Path -ChildPath "firefox.exe"
    if (Test-Path $firefoxPath) {
        $installedVersion = (Get-Item $firefoxPath).VersionInfo.ProductVersion
        $browsersList += ("Mozilla Firefox .......: {0}" -f $installedVersion)
    }
    $result = $browsersList -join "`n"

    Write-Output $result
}

function Get-ManagementStatus {
    $status = "Not managed"

    # Check for Zoho Endpoint Central agent service
    $uemsService = Get-Service -Name "ManageEngine UEMS - Agent" -ErrorAction SilentlyContinue
    if ($uemsService -and $uemsService.Status -eq "Running") {
        $status = "Zoho Endpoint Central"
    }

    # Check for SCCM
    $sccmService = Get-Service -Name "SMS Agent Host" -ErrorAction SilentlyContinue
    if ($sccmService -and $sccmService.Status -eq "Running") {
        $status = "SCCM"
    }

    # Check for Intune
    $intuneService = Get-Service -Name "IntuneManagementExtension" -ErrorAction SilentlyContinue
    if ($intuneService -and $intuneService.Status -eq "Running") {
        switch ($status) {
            "Not managed" { 
                $status = "Intune"
            }
            default {
                $status += " + Intune"
            }
        }
    }

    return $status
}

function Get-DiskFreeSpace {
    try {
        $driveInfo = Get-CimInstance Win32_LogicalDisk -Filter "DriveType=3" | Select-Object DeviceID, @{Name = "Free"; Expression = { [math]::Round($_.FreeSpace / 1GB, 2) } }, @{Name = "Size"; Expression = { [math]::Round($_.Size / 1GB, 2) } }
        $result = "{0} {1} / {2} GB" -f $driveInfo.DeviceID, $driveInfo.Free, $driveInfo.Size
    }
    catch {
        $result = "n/a"
    }

    Write-Output $result    
}

function Get-UserDomain {
    Write-Output $env:USERDNSDOMAIN
}

function Get-DeviceName {
    Write-Output $env:COMPUTERNAME
}

function Get-DeviceModelInfo {
    if ($ComputerInfo) {
        $result = (@($ComputerInfo.CsManufacturer, $ComputerInfo.CsSystemFamily, $ComputerInfo.CsModel, $ComputerInfo.BiosSeralNumber ) | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join "`n"
    }
    else {
        $result = "n/a"
    }

    Write-Output $result
}

function Get-DeviceBIOSInfo {
    if ($ComputerInfo) {
        $result = (@($ComputerInfo.BiosManufacturer, $ComputerInfo.BiosCaption, $ComputerInfo.BiosVersion, ($ComputerInfo.BiosReleaseDate).ToString("yyyy-MM-dd HH:mm:ss")) | Where-Object { -not [string]::IsNullOrWhiteSpace($_) }) -join "`n"
    }
    else {
        $result = "n/a"
    }

    Write-Output $result
}
# information to display
# - label
# - data displayed
# - function to call
# - flag to trigger a refresh of data when the form is displayed
#   e.g.: IP addresses when moving between networks, application versions that can be updated without a computer restart, ...

$deviceInfos = @(
    [PSCustomObject]@{
        Label     = "Last boot time"
        Data      = "n/a"
        Action    = "Get-LastBootTime"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "User domain"
        Data      = "n/a"
        Action    = "Get-UserDomain"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "User identity"
        Data      = "n/a"
        Action    = "Get-UserIdentity"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "Device domain"
        Data      = "n/a"
        Action    = "Get-DeviceDomain"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "Device name"
        Data      = "n/a"
        Action    = "Get-DeviceName"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "IP Address(es)"
        Data      = "n/a"
        Action    = "Get-IPAddress"
        Updatable = $true
    }
    [PSCustomObject]@{
        Label     = "Device Management"
        Data      = "n/a"
        Action    = "Get-ManagementStatus"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "OS Information"
        Data      = "n/a"
        Action    = "Get-OsInfo"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "Installed Browsers"
        Data      = "n/a"
        Action    = "Get-BrowsersInfo"
        Updatable = $true
    }
    [PSCustomObject]@{
        Label     = "MS Office Information"
        Data      = "n/a"
        Action    = "Get-MSOfficeInfo"
        Updatable = $true
    }
    [PSCustomObject]@{
        Label     = "Disk(s) free space"
        Data      = "n/a"
        Action    = "Get-DiskFreeSpace"
        Updatable = $true
    }    
    [PSCustomObject]@{
        Label     = "BIOS information"
        Data      = "n/a"
        Action    = "Get-DeviceBIOSInfo"
        Updatable = $false
    }
    [PSCustomObject]@{
        Label     = "Model information"
        Data      = "n/a"
        Action    = "Get-DeviceModelInfo"
        Updatable = $false
    }
)

# XAML for the main window
[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
        Title="Device Information" 
		Height="400" 
		MaxWidth="500"
		SizeToContent="WidthAndHeight"
		ResizeMode = "NoResize"
        WindowStartupLocation="CenterScreen"
        WindowStyle="ThreeDBorderWindow"
    >
	<StackPanel Orientation="Vertical" HorizontalAlignment="Stretch" VerticalAlignment="Top" >
		<StackPanel Background="LightBlue" HorizontalAlignment="Stretch" >
            <DataGrid 
                Name="DeviceInformation"                                        
                AutoGenerateColumns="False"
                CanUserAddRows="False"
                CanUserReorderColumns="False"
                CanUserSortColumns="False"
                IsReadOnly="True"
                SelectionMode="Single"
                Margin="5"
                GridLinesVisibility="None"
                HeadersVisibility="None"
            >                         
                <DataGrid.Columns>
                    <DataGridTextColumn Binding="{Binding Label}" Width="Auto" FontWeight="Bold" FontFamily="Arial" />
                    <DataGridTextColumn Binding="{Binding Data}" Width="*" FontFamily="Consolas, Courier New" />
                </DataGrid.Columns>
            </DataGrid>
		</StackPanel>
		<StackPanel HorizontalAlignment="Stretch" Background="LightBlue">
			<Button Name="CopyButton" HorizontalAlignment="Stretch" Content="Copy to Clipboard" />
			<Button Name="CloseButton" HorizontalAlignment="Stretch" Content="Close" />
		</StackPanel>
	</StackPanel>
</Window>
"@

[void][System.Reflection.Assembly]::LoadWithPartialName('PresentationFramework')

$okForm = $false
$xamlReader = (New-Object System.Xml.XmlNodeReader $xaml)
try {
    $mainForm = [Windows.Markup.XamlReader]::Load($xamlReader)
    $okForm = $true
}
catch [System.Management.Automation.MethodInvocationException] {
    Write-Warning "Problem occured when loading the XAML code.  Check the syntax for this control..."
    Write-Error $error[0].Exception.Message
}
catch {
    Write-Warning "Unable to load Windows.Markup.XamlReader. Double-check syntax."
    Write-Error $error[0].Exception.Message
}

if (-not $okForm) {
    return
}

# auto-creation of functions associated to named object in the XAML form
$xaml.SelectNodes("//*[@Name]") | ForEach-Object { Set-Variable -Name ($_.Name) -Value $mainForm.FindName($_.Name) }

# Hide window on startup, and don't show in the taskbar when displayed
$mainForm.ShowInTaskbar = $false
$mainForm.Hide()

# calculate the intitial values
foreach ($deviceInfo in $deviceInfos) {
    $deviceInfo.Data = & ($deviceInfo.Action)
}

$DeviceInformation.ItemsSource = $deviceInfos

# Create system tray icon
$appContext = New-Object System.Windows.Forms.ApplicationContext
$notifyIcon = New-Object System.Windows.Forms.NotifyIcon

# Use PowerShell's own icon (or specify a .ico file)
$iconFile = Join-Path -Path $PSScriptRoot -ChildPath "SysTray.ico"
if (-not (Test-Path -Path $iconFile)) {
    $iconFile = [System.Drawing.Icon]::ExtractAssociatedIcon("$env:SystemRoot\system32\WindowsPowerShell\v1.0\powershell.exe")
}

$notifyIcon.Icon = $iconFile
$notifyIcon.Text = "Device Information"
$notifyIcon.Visible = $true

# Context menu (right-click)
$contextMenu = New-Object System.Windows.Forms.ContextMenuStrip
$showItem = New-Object System.Windows.Forms.ToolStripMenuItem
$showItem.Text = "Show"
$exitItem = New-Object System.Windows.Forms.ToolStripMenuItem
$exitItem.Text = "Exit"
$contextMenu.Items.Add($showItem) | Out-Null
$contextMenu.Items.Add($exitItem) | Out-Null 
$notifyIcon.ContextMenuStrip = $contextMenu

# Convert NotifyIcon icon to WPF BitmapSource
function Convert-IconToImageSource($icon) {
    $bitmap = $icon.ToBitmap()
    $stream = New-Object System.IO.MemoryStream
    $bitmap.Save($stream, [System.Drawing.Imaging.ImageFormat]::Png)
    $stream.Position = 0
    $bitmapImage = New-Object System.Windows.Media.Imaging.BitmapImage
    $bitmapImage.BeginInit()
    $bitmapImage.StreamSource = $stream
    $bitmapImage.EndInit()
    $bitmapImage.Freeze()
    return $bitmapImage
}

# Set the image source (use the same icon as NotifyIcon)
$mainForm.Icon = Convert-IconToImageSource($notifyIcon.Icon)

# Left-click: toggle window visibility
$notifyIcon.Add_MouseClick({
        if ($_.Button -eq "Left") {
            if ($mainForm.IsVisible) {
                $mainForm.Hide()
            }
            else {
                $mainForm.Show()
                $mainForm.Activate()
            }
        }
    })

# Show menu item: show window
$showItem.Add_Click({
        $mainForm.Show()
        $mainForm.Activate()
    })

# actions on the form

# Close button hides the window, doesn't close it
$CloseButton.Add_Click(
    {
        $mainForm.Hide()
    }
)
# handle the [x] button on the form to hide the window
$mainForm.Add_Closing(
    {
        $_.Cancel = $true
        $mainForm.Hide()
    }
)

# refresh updatable data when the form is activated from the taskbar
$mainForm.Add_Activated({
        foreach ($deviceInfo in $deviceInfos) {
            if ($deviceInfo.Updatable) {
                $deviceInfo.Data = & ($deviceInfo.Action)
            }
        }
        $DeviceInformation.Items.Refresh()
    })

# Copy button: copy all info to clipboard
$CopyButton.Add_Click(
    {
        $text = $deviceInfos | Select-Object Label, Data | Format-Table -Wrap -AutoSize -HideTableHeaders | Out-String
        Set-Clipboard -Value $text        
        [System.Windows.MessageBox]::Show("Information copied to clipboard.", "Copied", "OK", "Information")
    }
)

# actions on the taskbar icon

# "exit" item: clean shutdown
$exitItem.Add_Click(
    {
        $notifyIcon.Visible = $false
        $notifyIcon.Dispose()
        $mainForm.Close()
        $appContext.ExitThread()
        # Release mutex and exit
        try { 
            $mutex.ReleaseMutex() 
        } 
        catch {}
        $mutex.Dispose()

    }
)

# Run the application context
[void][System.Windows.Forms.Application]::Run($appContext)
